
"use client";

import { useEffect, useState, Fragment as ReactFragment, Suspense } from 'react';
import { getBlogPostBySlug, submitBlogPostComment, getBlogPosts } from '@/services/apiService';
import { notFound, useParams } from 'next/navigation';
import Image from 'next/image';
import Link from 'next/link';
import { Badge } from '@/components/ui/badge';
import { CalendarDays, UserCircle, Tag, ThumbsUp, Send, Star as StarIconLucide, AlertCircle, Eye, BookText, Lightbulb } from 'lucide-react';
import { personalInfo, type CommentResource, type BlogPost, type NewCommentPayload, type SimplePostInfo, type SuggestedPost } from '@/lib/data';
import { AdSensePlaceholder } from '@/components/AdSensePlaceholder';
import { ShareButtons } from '@/components/blog/ShareButtons';
import { Button } from '@/components/ui/button';
import { Textarea } from '@/components/ui/textarea';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Skeleton } from '@/components/ui/skeleton';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { useToast } from '@/hooks/use-toast';
import { suggestRelatedPosts, type SuggestRelatedPostsInput } from '@/ai/flows/suggest-related-posts-flow';
import { BlogPostCard } from '@/components/blog/BlogPostCard';


// Props type is used by generateMetadata (which is in metadata.ts)
// This page component itself doesn't need Props if slug comes from useParams
type Props = {
  params: { slug: string };
};


const CommentDisplay = ({ comment }: { comment: CommentResource }) => {
  const handleLikeComment = (commentId: string | number) => {
    console.log(`Liking comment ${commentId} - API call would go here`);
    // Implement API call to like comment and update UI
  };

  const handleAddReaction = (commentId: string | number, reaction: string) => {
    console.log(`Adding reaction "${reaction}" to comment ${commentId} - API call would go here`);
    // Implement API call to add reaction and update UI
  };

  return (
    <Card className="mb-4 bg-muted/50">
      <CardHeader className="pb-2">
        <div className="flex items-center justify-between">
          <p className="font-semibold text-sm text-foreground">{comment.user?.name || comment.name || 'Anonymous'}</p>
          <p className="text-xs text-muted-foreground">
            {comment.created_at ? new Date(comment.created_at).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'Some time ago'}
          </p>
        </div>
      </CardHeader>
      <CardContent>
        <p className="text-sm text-muted-foreground">{comment.comment}</p>
      </CardContent>
      <CardFooter className="flex items-center justify-start gap-4 pt-2">
        <Button variant="ghost" size="sm" onClick={() => handleLikeComment(comment.id)} className="text-muted-foreground hover:text-primary">
          <ThumbsUp className="mr-1.5 h-4 w-4" /> Like ({comment.likes || 0})
        </Button>
        <div className="flex gap-1">
          {['😀', '❤️', '👍'].map(emoji => (
            <Button key={emoji} variant="ghost" size="icon" className="h-7 w-7 text-muted-foreground hover:scale-125 transition-transform" onClick={() => handleAddReaction(comment.id, emoji)}>
              {emoji}
            </Button>
          ))}
        </div>
      </CardFooter>
    </Card>
  );
};

export default function BlogPostPage() {
  const routeParams = useParams();
  const [post, setPost] = useState<BlogPost | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const { toast } = useToast();
  
  const [relatedPosts, setRelatedPosts] = useState<SuggestedPost[]>([]);
  const [isLoadingRelatedPosts, setIsLoadingRelatedPosts] = useState(false);
  const [relatedPostsError, setRelatedPostsError] = useState<string | null>(null);
  const [allPostsForRelated, setAllPostsForRelated] = useState<BlogPost[]>([]);


  const slug = typeof routeParams.slug === 'string' ? routeParams.slug : Array.isArray(routeParams.slug) ? routeParams.slug[0] : '';

  useEffect(() => {
    if (!slug) {
      setError("Post slug not found in URL.");
      setIsLoading(false);
      return;
    }

    async function fetchPostAndPrepareRelated() {
      setIsLoading(true);
      setError(null);
      const postResponse = await getBlogPostBySlug(slug);
      if (postResponse.success && postResponse.data) {
        setPost(postResponse.data);
        // Fetch all posts for related articles suggestion, once the current post is loaded
        const allPostsResponse = await getBlogPosts();
        if (allPostsResponse.success && allPostsResponse.data) {
          setAllPostsForRelated(allPostsResponse.data);
        } else {
          console.warn("Could not fetch all posts for related suggestions:", allPostsResponse.error);
          setAllPostsForRelated([]); // Proceed without related posts if this fails
        }
      } else {
        let uiErrorMessage = postResponse.error || `Failed to load post: ${slug}`;
        if (postResponse.statusCode === 404) {
            uiErrorMessage = `Blog post with slug "${slug}" not found.`;
            notFound(); 
            return; 
        } else if (postResponse.statusCode) {
            uiErrorMessage = `Failed to load post (Error ${postResponse.statusCode}). Please try again later.`;
        } else if (postResponse.statusCode === null && uiErrorMessage.toLowerCase().includes("failed to fetch")) { 
            uiErrorMessage = `Could not connect to the server to load the post. Please check your network and try again.`;
        }
        setError(uiErrorMessage);
        console.error(`BlogPostPage: Error fetching post "${slug}". Message: ${uiErrorMessage}. Status: ${postResponse.statusCode}. Full Response:`, postResponse);
      }
      setIsLoading(false);
    }
    fetchPostAndPrepareRelated();
  }, [slug]);

  useEffect(() => {
    if (post && allPostsForRelated.length > 0) {
      async function fetchRelated() {
        setIsLoadingRelatedPosts(true);
        setRelatedPostsError(null);
        try {
          const otherPosts: SimplePostInfo[] = allPostsForRelated
            .filter(p => p.slug !== post.slug) // Exclude current post
            .map(p => ({
              title: p.frontmatter.title,
              slug: p.slug,
              tags: p.frontmatter.tags || [],
              category: p.frontmatter.category,
              excerpt: p.frontmatter.excerpt,
            }));

          if (otherPosts.length === 0) {
            setIsLoadingRelatedPosts(false);
            return; // No other posts to suggest from
          }
          
          const flowInput: SuggestRelatedPostsInput = {
            currentPostTitle: post.frontmatter.title,
            currentPostTags: post.frontmatter.tags || [],
            currentPostCategory: post.frontmatter.category,
            currentPostExcerpt: post.frontmatter.excerpt.substring(0, 500), // Limit excerpt length
            allPosts: otherPosts,
          };
          const result = await suggestRelatedPosts(flowInput);
          setRelatedPosts(result.relatedPosts || []);
        } catch (e) {
          console.error("Error fetching related posts:", e);
          setRelatedPostsError("Could not load related articles.");
          setRelatedPosts([]);
        } finally {
          setIsLoadingRelatedPosts(false);
        }
      }
      fetchRelated();
    }
  }, [post, allPostsForRelated]);


  if (isLoading) {
    return (
      <div className="prose prose-lg dark:prose-invert max-w-none mx-auto p-6 sm:p-8">
        <Skeleton className="h-96 w-full mb-8 rounded-lg" />
        <Skeleton className="h-12 w-3/4 mx-auto mb-4" />
        <Skeleton className="h-6 w-1/2 mx-auto mb-8" />
        <div className="space-y-4">
          {[...Array(5)].map((_, i) => (
            <ReactFragment key={i}>
              <Skeleton className="h-4 w-full" />
              <Skeleton className="h-4 w-5/6" />
              <Skeleton className="h-4 w-3/4" />
            </ReactFragment>
          ))}
        </div>
      </div>
    );
  }

  if (error && !post) { 
    return (
      <Alert variant="destructive" className="max-w-2xl mx-auto my-8">
        <AlertCircle className="h-4 w-4" />
        <AlertTitle>Error Loading Post</AlertTitle>
        <AlertDescription>{error}</AlertDescription>
      </Alert>
    );
  }
  
  if (!post) {
    // This case might be hit if notFound() was called, or if API returned success:false but no specific error message was set.
    return (
        <Alert variant="default" className="max-w-2xl mx-auto my-8">
            <AlertCircle className="h-4 w-4" />
            <AlertTitle>Post Not Available</AlertTitle>
            <AlertDescription>The requested blog post could not be displayed. It might have been moved or the provided link is incorrect.</AlertDescription>
        </Alert>
    );
  }

  const contentHtml = post.content;
  
  const displayDate = post.frontmatter.date
    ? new Date(post.frontmatter.date).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' })
    : "Date not available";
  

  const handleLikePost = () => {
    console.log(`Liking post "${post.slug}" - API call would go here`);
    // Example: setPost(prev => prev ? { ...prev, frontmatter: { ...prev.frontmatter, likes: (prev.frontmatter.likes || 0) + 1 } } : null);
  };

  const handleCommentSubmit = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!post || !post.id) {
      toast({ title: "Error", description: "Post ID is missing.", variant: "destructive" });
      return;
    }

    const formData = new FormData(event.currentTarget);
    const newCommentPayload: NewCommentPayload = {
      name: formData.get('commenterName') as string,
      email: formData.get('commenterEmail') as string, 
      comment: formData.get('commentText') as string,
    };

    const response = await submitBlogPostComment(post.id, newCommentPayload);

    if (response.success && response.data) {
      toast({
        title: "Comment Submitted!",
        description: response.error || "Your comment has been submitted successfully.", // API might return message in 'error' field for success
        variant: "default",
      });
      event.currentTarget.reset();
      setPost(prevPost => {
        if (!prevPost) return null;
        const newComments = [...(prevPost.comments || []), response.data!]; // Optimistically add new comment
        return {
          ...prevPost,
          comments: newComments,
          frontmatter: { // Also update comment count
            ...prevPost.frontmatter,
            comments_count: newComments.length,
          }
        };
      });
      console.log("New comment added, UI should update:", response.data);
    } else {
      toast({
        title: "Error Submitting Comment",
        description: response.error || "An unexpected error occurred.",
        variant: "destructive",
      });
    }
  };

  return (
    <article className="prose prose-lg dark:prose-invert max-w-none mx-auto bg-card p-6 sm:p-8 rounded-xl shadow-xl">
      {post.frontmatter.image && (
        <div className="relative w-full h-64 md:h-96 mb-8 rounded-lg overflow-hidden shadow-md">
          <Image
            src={post.frontmatter.image}
            alt={post.frontmatter.title}
            fill
            style={{ objectFit: 'cover' }}
            priority
            sizes="(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"
            data-ai-hint={post.frontmatter.dataAiHint || 'blog post header'}
          />
        </div>
      )}
      <header className="mb-8 text-center">
        <h1 className="text-4xl md:text-5xl font-extrabold text-primary mb-3">
          {post.frontmatter.title}
        </h1>
        <div className="flex flex-wrap justify-center items-center space-x-4 text-muted-foreground text-sm">
          <span className="flex items-center"><CalendarDays className="mr-1.5 h-4 w-4" /> {displayDate}</span>
          <span className="flex items-center"><UserCircle className="mr-1.5 h-4 w-4" /> By {post.frontmatter.author}</span>
          {post.frontmatter.category && <span className="flex items-center"><Tag className="mr-1.5 h-4 w-4" /> {post.frontmatter.category}</span>}
        </div>
        {post.frontmatter.tags && post.frontmatter.tags.length > 0 && (
          <div className="mt-4 flex flex-wrap justify-center gap-2">
            {post.frontmatter.tags.map(tag => (
              <Badge key={tag} variant="secondary" className="bg-accent/10 text-accent hover:bg-accent/20">
                {tag}
              </Badge>
            ))}
          </div>
        )}
      </header>

      {post.frontmatter.showTopAd && <AdSensePlaceholder slot={`Blog Post (${post.slug}) - Top`} />}

      <div dangerouslySetInnerHTML={{ __html: contentHtml }} />

      {post.frontmatter.affiliateProducts && post.frontmatter.affiliateProducts.length > 0 && (
        <section className="mt-10 pt-6 border-t">
          <h2 className="text-2xl font-semibold mb-6 text-center">Featured Products</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {post.frontmatter.affiliateProducts.map((product, index) => (
               <Card key={index} className="my-4 p-4 border rounded-lg shadow-md bg-muted/30">
                <CardHeader className="p-0 pb-2">
                  <CardTitle className="text-lg text-primary">{product.name}</CardTitle>
                </CardHeader>
                <CardContent className="p-0">
                  {product.image && (
                    <div className="relative w-full h-40 mb-3 rounded overflow-hidden">
                      <Image src={product.image} alt={product.name} fill style={{objectFit: 'cover'}} data-ai-hint={product.dataAiHint || 'product image'} sizes="400px" />
                    </div>
                  )}
                  <p className="text-sm text-muted-foreground mb-2">{product.review}</p>
                  {product.rating && (
                    <div className="flex items-center mb-2">
                      {[...Array(5)].map((_, i) => (
                        <StarIconLucide key={i} className={`h-4 w-4 ${i < product.rating ? 'text-yellow-400 fill-yellow-400' : 'text-muted-foreground/30'}`} />
                      ))}
                      <span className="ml-1.5 text-xs text-muted-foreground">({product.rating}/5)</span>
                    </div>
                  )}
                  <Link href={product.link} target="_blank" rel="noopener noreferrer sponsored">
                    <Button variant="default" size="sm" className="w-full">
                      Check Price & Details
                    </Button>
                  </Link>
                  <p className="text-xs text-center mt-1.5 text-muted-foreground/70">(Affiliate Link)</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </section>
      )}

      <section className="mt-10 pt-6 border-t flex flex-col sm:flex-row justify-between items-center gap-4">
        <div className="flex items-center gap-2">
            <Button variant="outline" onClick={handleLikePost} className="group">
            <ThumbsUp className="mr-2 h-5 w-5 text-primary group-hover:animate-pulse" />
            Like ({post.frontmatter.likes || 0})
            </Button>
            <Button variant="ghost" className="group text-muted-foreground" disabled> {/* Placeholder for views */}
                <Eye className="mr-2 h-5 w-5" />
                Views ({post.frontmatter.views_count || 0})
            </Button>
        </div>
        <ShareButtons postUrl={`${personalInfo.siteUrl}/blog/${post.slug}`} postTitle={post.frontmatter.title} />
      </section>

      {/* Related Articles Section */}
      <section className="mt-10 pt-6 border-t">
        <h2 className="text-2xl font-semibold mb-6 flex items-center">
          <Lightbulb className="mr-2 h-6 w-6 text-primary" />
          Related Articles
        </h2>
        {isLoadingRelatedPosts && (
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
            {[...Array(2)].map((_, i) => (
              <Card key={i} className="p-4">
                <Skeleton className="h-5 w-3/4 mb-2" />
                <Skeleton className="h-3 w-1/2 mb-3" />
                <Skeleton className="h-8 w-24 rounded" />
              </Card>
            ))}
          </div>
        )}
        {!isLoadingRelatedPosts && relatedPostsError && (
          <Alert variant="destructive">
            <AlertCircle className="h-4 w-4" />
            <AlertTitle>Error</AlertTitle>
            <AlertDescription>{relatedPostsError}</AlertDescription>
          </Alert>
        )}
        {!isLoadingRelatedPosts && !relatedPostsError && relatedPosts.length > 0 && (
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6 not-prose"> {/* Added not-prose to prevent prose styling on cards */}
            {relatedPosts.map((relatedPost, index) => {
              // Find the full BlogPost object from allPostsForRelated to pass to BlogPostCard
              const fullRelatedPost = allPostsForRelated.find(p => p.slug === relatedPost.slug);
              return fullRelatedPost ? (
                <BlogPostCard key={relatedPost.slug || index} post={fullRelatedPost} viewMode="grid" />
              ) : (
                // Fallback if full post details aren't found (should ideally not happen)
                <Card key={`simple-${relatedPost.slug || index}`} className="p-4 shadow-md">
                  <CardTitle className="text-lg mb-2 hover:text-primary">
                    <Link href={`/blog/${relatedPost.slug}`}>{relatedPost.title}</Link>
                  </CardTitle>
                  <Link href={`/blog/${relatedPost.slug}`}>
                    <Button variant="link" className="p-0 h-auto text-sm">Read more</Button>
                  </Link>
                </Card>
              );
            })}
          </div>
        )}
        {!isLoadingRelatedPosts && !relatedPostsError && relatedPosts.length === 0 && !isLoading && (
          <p className="text-muted-foreground">No related articles found.</p>
        )}
      </section>


      <section className="mt-10 pt-6 border-t">
        <h2 className="text-2xl font-semibold mb-6">Comments ({post.frontmatter.comments_count || post.comments?.length || 0})</h2>
        {post.comments && post.comments.length > 0 ? (
          <div className="space-y-6">
            {post.comments.map((comment) => (
              <CommentDisplay key={comment.id} comment={comment} />
            ))}
          </div>
        ) : (
          <p className="text-muted-foreground">No comments yet. Be the first to comment!</p>
        )}

        <form onSubmit={handleCommentSubmit} className="mt-8 space-y-4 p-6 border rounded-lg shadow-sm bg-background">
          <h3 className="text-lg font-semibold">Leave a Reply</h3>
          <div>
            <label htmlFor="commenterName" className="block text-sm font-medium text-foreground mb-1">Name</label>
            <Input type="text" id="commenterName" name="commenterName" required className="w-full" />
          </div>
          <div>
            <label htmlFor="commenterEmail" className="block text-sm font-medium text-foreground mb-1">Email (won't be published)</label>
            <Input type="email" id="commenterEmail" name="commenterEmail" required className="w-full" />
          </div>
          <div>
            <label htmlFor="commentText" className="block text-sm font-medium text-foreground mb-1">Comment</label>
            <Textarea id="commentText" name="commentText" rows={4} required className="w-full" />
          </div>
          <Button type="submit" variant="default">
            <Send className="mr-2 h-4 w-4" /> Post Comment
          </Button>
        </form>
      </section>

      {post.frontmatter.showBottomAd && <AdSensePlaceholder slot={`Blog Post (${post.slug}) - Bottom`} className="mt-8" />}

      <div className="mt-12 pt-8 border-t text-center">
        <Link href="/blog" className="text-primary hover:underline font-medium">
          &larr; Back to Blog
        </Link>
      </div>
    </article>
  );
}
